﻿using System.Text.RegularExpressions;
using Microsoft.ProgramSynthesis.NLToM.Tests;
using Microsoft.ProgramSynthesis.Utils;
using Microsoft.ProgramSynthesis.Wrangling.Schema.TableOutput;
using Newtonsoft.Json;
using static NLToCode.Evaluation.Bing21Utils;
using static NLToCode.Evaluation.Utils;

namespace NLToCode.Evaluation {
    public static class Logger {
        public enum Priority {
            Trace = 0,
            Debug = 1,
            Info = 2,
            Warn = 3,
            Error = 4,
            Fatal = 5
        };

        public static Priority LogLevel { get; set; } = Priority.Info;

        public enum ParseMode { File, Arguments };

        private static void Log(Priority priority, string linePrefix, string message) {
            if (priority >= LogLevel) {
                foreach (var line in message.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None)) {
                    Console.WriteLine($"{linePrefix}{line}");
                }
            }
        }

        public static void Trace(string message) {
            Log(Priority.Trace, "TRACE: ", message);
        }

        public static void Debug(string message) {
            Log(Priority.Debug, "DEBUG: ", message);
        }

        public static void Info(string message) {
            Log(Priority.Info, "INFO: ", message);
        }

        public static void Warn(string message) {
            Log(Priority.Warn, "WARN: ", message);
        }

        public static void Error(string message) {
            Log(Priority.Error, "ERROR: ", message);
        }

        public static void Fatal(string message) {
            Log(Priority.Fatal, "FATAL: ", message);
        }
    }
    internal class Driver {
        static async Task Main(string[] args) {
            Configuration configuration;
            try {
                Console.WriteLine($"SystemLog: ParsingArguments");
                configuration = Configuration.Parse(args);
                Logger.LogLevel = configuration.LogLevel;
                Logger.Info($"Setting log level to {configuration.LogLevel}.");
            }
            catch (JsonSerializationException) {
                Logger.Fatal($"Unable to parse configuration!");
                return;
            }

            Console.WriteLine($"SystemLog: In {configuration.Mode} mode.");
            Logger.Info($"Running modes: {configuration.Mode}");
            Logger.Info($"In {configuration.Mode} mode.");
            try {
                switch (configuration.Mode) {
                    case Configuration.OperationMode.Execute:
                        await Execute(configuration);
                        break;
                    case Configuration.OperationMode.Metrics:
                        Metrics(configuration);
                        break;
                    case Configuration.OperationMode.Unroll:
                        Unroll(configuration);
                        break;
                    default:
                        throw new NotImplementedException();
                }
            }
            catch (Exception e) {
                Console.WriteLine($"SystemLog: Error in prose module:- {e.PrettyPrint()} ");
                Logger.Error($"Exception: {e}");
            }
        }

        private static async Task Execute(Configuration configuration) {
            Console.WriteLine($"SystemLog: Execute");
            var inputPath = configuration.InputPath.Trim();
            Console.WriteLine(inputPath);
            var outputPath = configuration.OutputPath.Trim();
            var fileType = configuration.FileType;
            if (!Directory.Exists(outputPath)) Directory.CreateDirectory(outputPath);
            Logger.Info($"Executing programs for files in the folder: {inputPath}.");
            var benchmarks = Benchmark.ParseMany(inputPath, fileType.ToString());
            //Execute code on benchmarks
            List<Benchmark> outBenchmarks = await ExecuteBenchmarks(benchmarks, configuration);
            foreach(var (idx, output) in outBenchmarks.Enumerate()) {
                File.WriteAllText(outputPath + $"/{output.Id}_{idx}.json", JsonConvert.SerializeObject(output));
            }
        }

        private static void Metrics(Configuration configuration) {
            Console.WriteLine($"SystemLog: Metrics");
            var inputPath = configuration.InputPath;
            var outputPath = configuration.OutputPath;
            var fileType = configuration.FileType;
            Logger.Info($"Plotting metrics for executed file for files in the folder: {inputPath}.");
            var benchmarks = Benchmark.ParseMany(inputPath, fileType.ToString());
            Logger.Info($"Found {benchmarks.Count} files.");
            var metrics = PlotMetrics(benchmarks, configuration);
            Logger.Info($"Stored results in {outputPath}");
            File.WriteAllText(outputPath, metrics);
        }

        private static void Unroll(Configuration configuration) {
            Console.WriteLine($"SystemLog: Unroll");
            var inputPath = configuration.InputPath;
            var outputPath = configuration.OutputPath;
            var fileType = configuration.FileType;
            int stepsEmpty = 0;
            int parsingProblem = 0;
            Logger.Info($"Unroll: {inputPath}");
            var benchmarks = Benchmark.ParseMany(inputPath, fileType.ToString());
            Logger.Info($"Found {benchmarks.Count} files");
            using (StreamWriter sb = new StreamWriter(outputPath, false)) {
                foreach (var benchmark in benchmarks) {
                    benchmark.MetaData = new StepsMetadata() {Filepath = benchmark.MetaData.Filepath ,Steps = UnrollQuery(benchmark.Answer) };
                    if (benchmark.MetaData.Steps == null) {
                        parsingProblem++;
                    }
                    else if (benchmark.MetaData.Steps.Count == 0) {
                        stepsEmpty++;
                        Console.WriteLine(benchmark.Answer);
                    }
                    else {
                        sb.WriteLine(JsonConvert.SerializeObject(benchmark));
                    }
                }
            }
            Console.WriteLine($"Filed={parsingProblem}\nZeroSteps={stepsEmpty}");
        }
    }
}